// ==========================================================================
//
// = LIBRARY
//     PCIEFPGA
// 
// = FILENAME
//     PCIEMemInfo.cpp
// 
// = COPYRIGHT
//     (C) Copyright 2007 Agilent Technologies
//
// ==========================================================================
#include "PCIEMemInfo.h"

// ATTENTION: Do not modify order without changing order in header file
const CPCIEMemInfoProperty CPCIEMemInfo::mPropTable[] = {

//   mPropNum                          mDefault mDwNum mBitSize mBitPos

    {BLOCKMEM_R31,                      0x00000000, 0, 1,  31},
    {BLOCKMEM_FMT,                      0x00000000, 0, 2,  29},
    {BLOCKMEM_TYPE,                     0x00000000, 0, 5,  24},
    {BLOCKMEM_R23,                      0x00000000, 0, 1,  23},
    {BLOCKMEM_TC,                       0x00000000, 0, 3,  20},
    {BLOCKMEM_R19_16,                   0x00000000, 0, 4,  16},
    {BLOCKMEM_ATTR,                     0x00000000, 0, 2,  14},
    {BLOCKMEM_R11_10,                   0x00000000, 0, 2,  12},
    {BLOCKMEM_MARK,                     0x00000000, 0, 4,   8}, //?
    {BLOCKMEM_REPEAT_NEXT,              0x00000000, 0, 8,   0},
    {BLOCKMEM_ADDR_LO,                  0x00000000, 1,30,   2},
    {BLOCKMEM_DATA_IND,                 0x00000000, 1, 1,   1},
    {BLOCKMEM_DATA_COMP,                0x00000000, 1, 1,   0},
    {BLOCKMEM_ADDR_HI,                  0x00000000, 2,32,   0},
    {BLOCKMEM_INT_ADDR_12_2,            0x00000000, 3,11,  21},
    {BLOCKMEM_LENGTH,                   0x00000001, 3,21,   0},


    {BLOCKMEM_PATT,                     0x00000000, 4, 4,  20}, // Used pattern terms (bitmap) for wait/repeat until
    {BLOCKMEM_WAITREPEAT,               0x00000000, 4, 3,  24}, // Determines wait/repeat behavior
    
    {BLOCKMEM_ADDR_IS_64,               0x00000000, 4, 1,  19},
    {BLOCKMEM_FCT_NUM,                  0x00000000, 4, 3,  16},
    {BLOCKMEM_FDW_BE,                   0x0000000f, 4, 4,  12},
    {BLOCKMEM_LDW_BE,                   0x0000000f, 4, 4,   8},
    {BLOCKMEM_INT_ADDR_20_13,           0x00000000, 4, 8,   0},
    {BLOCKMEM_ADDR_LO_REGNUM_BITS,      0x00000000, 1,6,   2},
    {BLOCKMEM_ADDR_LO_EXTREGNUM_BITS,   0x00000000, 1,4,   8},
    {BLOCKMEM_ADDR_LO_RSVD107_BITS,     0x00000000, 1,4,  12},
    {BLOCKMEM_ADDR_LO_FUNCNUM_BITS,     0x00000000, 1,3,  16},
    {BLOCKMEM_ADDR_LO_DEVNUM_BITS,      0x00000000, 1,5,  19},
    {BLOCKMEM_ADDR_LO_BUSNUM_BITS,      0x00000000, 1,8,  24},
    {BLOCKMEM_FDW_LDW_BE_MCODE,         0x00000000, 4,8,   8},
    {BLOCKMEM_DATAIND_COMP,             0x00000000, 1,2,   0},
    {REQBEHMEM_GAP_12_0,                0x00000000, 0,13,  19},
    {REQBEHMEM_LENGTH,                  0x00000001, 0,11,   8},
    {REQBEHMEM_REPEAT_NEXT,             0x00000000, 0, 8,   0},
    {REQBEHMEM_INCORRECT_LCRC,          0x00000000, 1, 1,  23},
    {REQBEHMEM_NULLIFIED_TLP,           0x00000000, 1, 1,  22},
    {REQBEHMEM_WRONG_PYLD_SIZE,         0x00000000, 1, 1,  21},
    {REQBEHMEM_INSERT_TD,               0x00000000, 1, 1,  19},
    {REQBEHMEM_INCORRECT_ECRC,          0x00000000, 1, 1,  20},
    {REQBEHMEM_INCORRECT_DISP,          0x00000000, 1, 1,  18},
    {REQBEHMEM_EP,                      0x00000000, 1, 1,  17},
    {REQBEHMEM_TD,                      0x00000000, 1, 1,  16},
    {REQBEHMEM_TAG,                     0x00000000, 1, 8,   8},
    {REQBEHMEM_SW_TAG,                  0x00000000, 1, 1,   7},
    {REQBEHMEM_PRIORITY,                0x00000000, 1, 4,   3},
    {REQBEHMEM_GAP_15_13,               0x00000000, 1, 3,   0},
    {REQBEHMEM_REPLACE_STP,             0x00000000, 1, 1,   25},
    {REQBEHMEM_REPLACE_END,             0x00000000, 1, 1,   26},
    {REQBEHMEM_REPLACE_SEQNO,           0x00000000, 1, 1,   24},
    {COMPBEHMEM_INCORRECT_LCRC,         0x00000000, 0, 1,  28}, 
    {COMPBEHMEM_NULLIFIED_TLP,          0x00000000, 0, 1,  27},
    {COMPBEHMEM_WRONG_PYLD_SIZE,        0x00000000, 0, 1,  26},
    {COMPBEHMEM_INSERT_TD,              0x00000000, 0, 1,  24},
    {COMPBEHMEM_INCORRECT_ECRC,         0x00000000, 0, 1,  25},
    {COMPBEHMEM_INCORRECT_DISP,         0x00000000, 0, 1,  23},
    {COMPBEHMEM_EP,                     0x00000000, 0, 1,  22},
    {COMPBEHMEM_TD,                     0x00000000, 0, 1,  21},
    {COMPBEHMEM_PRIORITY,               0x00000001, 0, 4,  17},
    {COMPBEHMEM_COMPL_STATUS,           0x00000000, 0, 3,  14},
    {COMPBEHMEM_RCBCOUNT,               0x0000003F, 0, 6,   8},
    {COMPBEHMEM_REPEAT_NEXT,            0x00000000, 0, 8,   0},
    {COMPBEHMEM_REPLACE_STP,            0x00000000, 0, 1,   30},
    {COMPBEHMEM_REPLACE_END,            0x00000000, 0, 1,   31},
    {COMPBEHMEM_REPLACE_SEQNO,          0x00000000, 0, 1,   29},
    {COMPBEHMEM_DISCARD,                0x00000000, 1, 1,   0},
};

#define PROPTABLE_SIZE (sizeof (CPCIEMemInfo::mPropTable) / sizeof (CPCIEMemInfoProperty))

// ----------------------------------------------------------------------- 
CPCIEMemInfo::CPCIEMemInfo()
{  
}

// ----------------------------------------------------------------------- 
CPCIEMemInfo::~CPCIEMemInfo()
{
}

// ----------------------------------------------------------------------- 
// Block Memory
// ----------------------------------------------------------------------- 
UInt8 
CPCIEMemInfo::getBitPos(const EPCIEBlock& prop)
{
  return getPropertyInfo(prop).mBitPos;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIEMemInfo::getBitSize(const EPCIEBlock& prop)
{
  return getPropertyInfo(prop).mBitSize;
}

// ----------------------------------------------------------------------- 
UInt32 
CPCIEMemInfo::getDefault(const EPCIEBlock& prop)
{
  return getPropertyInfo(prop).mDefault;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIEMemInfo::getDWNum(const EPCIEBlock& prop)
{
  return getPropertyInfo(prop).mDwNum;
}

//---------------------------------------------------------------------------
void 
CPCIEMemInfo::setPropVal
(
 const EPCIEBlock& prop,
 UInt32& currentVal,
 const UInt32& newOffsetValue 
)
{
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitPos(prop);

  setBits(currentVal,bitPos,bitSize,newOffsetValue);
}

//---------------------------------------------------------------------------
void 
CPCIEMemInfo::setPropDefaultVal
(
 const EPCIEBlock& prop,
 UInt32& currentVal
)
{
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitPos(prop);
  UInt32 newOffsetValue = getDefault(prop);

  setBits(currentVal,bitPos,bitSize,newOffsetValue);
}


//---------------------------------------------------------------------------
UInt32
CPCIEMemInfo::getPropVal
(
 const EPCIEBlock& prop,
 const UInt32& currentVal
)
{
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitPos(prop);
  return getBits(currentVal,bitPos,bitSize);

}
// ----------------------------------------------------------------------- 
CPCIEMemInfoProperty 
CPCIEMemInfo::getPropertyInfo(const EPCIEBlock& prop)
{
  switch(prop)
  {
    case PCIE_BLOCK_LEN: return mPropTable[BLOCKMEM_LENGTH];
    case PCIE_BLOCK_RSVD007: return mPropTable[BLOCKMEM_R31];
    case PCIE_BLOCK_FMT: return mPropTable[BLOCKMEM_FMT];
    case PCIE_BLOCK_TYPE: return mPropTable[BLOCKMEM_TYPE];
    case PCIE_BLOCK_RSVD017: return mPropTable[BLOCKMEM_R23];
    case PCIE_BLOCK_TC: return mPropTable[BLOCKMEM_TC];
    case PCIE_BLOCK_RSVD013: return mPropTable[BLOCKMEM_R19_16];
    case PCIE_BLOCK_ATTR: return mPropTable[BLOCKMEM_ATTR];
    case PCIE_BLOCK_RSVD023: return mPropTable[BLOCKMEM_R11_10];
    case PCIE_BLOCK_1STDWBE: return mPropTable[BLOCKMEM_FDW_BE];
    case PCIE_BLOCK_LASTDWBE: return mPropTable[BLOCKMEM_LDW_BE];
    case PCIE_BLOCK_REPEAT: return mPropTable[BLOCKMEM_REPEAT_NEXT];  
    //    case PCIE_BLOCK_CFG_RSVD111: return mPropTable[];
    case PCIE_BLOCK_CFG_REGNUM: return mPropTable[BLOCKMEM_ADDR_LO_REGNUM_BITS];
    case PCIE_BLOCK_CFG_EXTREGNUM: return mPropTable[BLOCKMEM_ADDR_LO_EXTREGNUM_BITS];
    case PCIE_BLOCK_CFG_RSVD107: return mPropTable[BLOCKMEM_ADDR_LO_RSVD107_BITS];
    case PCIE_BLOCK_CFG_FUNCNUM: return mPropTable[BLOCKMEM_ADDR_LO_FUNCNUM_BITS];
    case PCIE_BLOCK_CFG_DEVNUM: return mPropTable[BLOCKMEM_ADDR_LO_DEVNUM_BITS];
    case PCIE_BLOCK_CFG_BUSNUM: return mPropTable[BLOCKMEM_ADDR_LO_BUSNUM_BITS];
    case PCIE_BLOCK_MEM64_ADDRHI: return mPropTable[BLOCKMEM_ADDR_HI];
    case PCIE_BLOCK_MEM64_ADDRLO: return mPropTable[BLOCKMEM_ADDR_LO];
//    case PCIE_BLOCK_MEM64_RSVD151: return mPropTable[];
    case PCIE_BLOCK_MEM32_ADDR: return mPropTable[BLOCKMEM_ADDR_LO];
//    case PCIE_BLOCK_MEM32_RSVD111: return mPropTable[];
//    case PCIE_BLOCK_MCODE: return mPropTable[BLOCKMEM_FDW_LDW_BE_MCODE];
    case PCIE_BLOCK_IO_ADDR: return mPropTable[BLOCKMEM_ADDR_LO];
//    case PCIE_BLOCK_IO_RSVD111: return mPropTable[];
//    case PCIE_BLOCK_MSGAS_RSVD047: return mPropTable[];
//    case PCIE_BLOCK_MSGAS_RSVD087: return mPropTable[];
//    case PCIE_BLOCK_MSGAS_RSVD127: return mPropTable[];        
//    case PCIE_BLOCK_MSG_BYTES08_11: return mPropTable[];        
//    case PCIE_BLOCK_MSG_BYTES12_15: return mPropTable[];    
//    case PCIE_BLOCK_MSGASD_RSVD047: return mPropTable[];  
//    case PCIE_BLOCK_MSGASD_RSVD087: return mPropTable[];    
//    case PCIE_BLOCK_MSGASD_RSVD127: return mPropTable[];   
//    case PCIE_BLOCK_TLPTYPE: return mPropTable[];   
    case PCIE_BLOCK_RESOURCE: return mPropTable[BLOCKMEM_DATAIND_COMP];   
    case PCIE_BLOCK_PATTERN_TERM: return mPropTable[BLOCKMEM_PATT];
    case PCIE_BLOCK_WAITREPEAT: return mPropTable[BLOCKMEM_WAITREPEAT];
    case PCIE_BLOCK_MARK: return mPropTable[BLOCKMEM_MARK];   
    default:
          AGT_THROW("Invalid CPCIEMemInfo property");
  }
}

//---------------------------------------------------------------------------
void
CPCIEMemInfo::intAddrWrite(const UInt32& val, UInt32& lowVal, UInt32& hiVal)
{
  writeSplitVal(val,lowVal,hiVal,BLOCKMEM_INT_ADDR_12_2,BLOCKMEM_INT_ADDR_20_13);
}

//---------------------------------------------------------------------------
UInt32 
CPCIEMemInfo::intAddrRead(const UInt32& lowVal, const UInt32& hiVal)
{
  return readSplitVal(lowVal,hiVal,BLOCKMEM_INT_ADDR_12_2,BLOCKMEM_INT_ADDR_20_13);
}
//---------------------------------------------------------------------------
void
CPCIEMemInfo::intAddrDefaultWrite(UInt32& lowVal, UInt32& hiVal)
{
  splitValWriteDefault(lowVal,hiVal,BLOCKMEM_INT_ADDR_12_2,BLOCKMEM_INT_ADDR_20_13);
}
//---------------------------------------------------------------------------
void
CPCIEMemInfo::setIs64Bit(UInt32& val,const bool& is64bit)
{
  UInt8 bitPos = mPropTable[BLOCKMEM_ADDR_IS_64].mBitPos;
  setBits(val,bitPos,1,is64bit);
}

//---------------------------------------------------------------------------
bool
CPCIEMemInfo::getIs64Bit(const UInt32& val)
{
  UInt8 bitPos = mPropTable[BLOCKMEM_ADDR_IS_64].mBitPos;
  return (getBits(val,bitPos,1) == 0x0 ? false : true);
}

// ----------------------------------------------------------------------- 
// Requester Behaviour Memory
// ----------------------------------------------------------------------- 
UInt8 
CPCIEMemInfo::getBitPos(const EPCIEReqBeh& prop)
{
  return getPropertyInfo(prop).mBitPos;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIEMemInfo::getBitSize(const EPCIEReqBeh& prop)
{
  return getPropertyInfo(prop).mBitSize;
}

// ----------------------------------------------------------------------- 
UInt32 
CPCIEMemInfo::getDefault(const EPCIEReqBeh& prop)
{
  return getPropertyInfo(prop).mDefault;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIEMemInfo::getDWNum(const EPCIEReqBeh& prop)
{
  return getPropertyInfo(prop).mDwNum;
}


//---------------------------------------------------------------------------
void 
CPCIEMemInfo::setPropVal
(
 const EPCIEReqBeh& prop,
 UInt32& currentVal,
 const UInt32& newOffsetValue 
)
{
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitPos(prop);

  setBits(currentVal,bitPos,bitSize,newOffsetValue);
}

//---------------------------------------------------------------------------
void 
CPCIEMemInfo::setPropDefaultVal
(
 const EPCIEReqBeh& prop,
 UInt32& currentVal
)
{
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitPos(prop);
  UInt32 newOffsetValue = getDefault(prop);

  setBits(currentVal,bitPos,bitSize,newOffsetValue);
}


//---------------------------------------------------------------------------
UInt32
CPCIEMemInfo::getPropVal
(
 const EPCIEReqBeh& prop,
 const UInt32& currentVal
)
{
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitPos(prop);
  return getBits(currentVal,bitPos,bitSize);

}
// ----------------------------------------------------------------------- 
CPCIEMemInfoProperty 
CPCIEMemInfo::getPropertyInfo(const EPCIEReqBeh& prop)
{
  switch(prop)
  {
  case PCIE_REQBEH_EP: return mPropTable[REQBEHMEM_EP];  
  case PCIE_REQBEH_TD: return mPropTable[REQBEHMEM_TD];    
  case PCIE_REQBEH_INCORRECT_LCRC: return mPropTable[REQBEHMEM_INCORRECT_LCRC];  
  case PCIE_REQBEH_NULLIFIED_TLP: return mPropTable[REQBEHMEM_NULLIFIED_TLP];  
  case PCIE_REQBEH_WRONG_PYLD_SIZE: return mPropTable[REQBEHMEM_WRONG_PYLD_SIZE]; 
  case PCIE_REQBEH_INSERT_TD: return mPropTable[REQBEHMEM_INSERT_TD];       
  case PCIE_REQBEH_PRIORITY: return mPropTable[REQBEHMEM_PRIORITY];        
  case PCIE_REQBEH_INCORRECT_ECRC: return mPropTable[REQBEHMEM_INCORRECT_ECRC];  
  case PCIE_REQBEH_INCORRECT_DISP: return mPropTable[REQBEHMEM_INCORRECT_DISP];  
  case PCIE_REQBEH_AUTOTAG: return mPropTable[REQBEHMEM_SW_TAG];         
  case PCIE_REQBEH_TAG: return mPropTable[REQBEHMEM_TAG];            
    //case PCIE_REQBEH_GAP: return mPropTable[];            
  case PCIE_REQBEH_REPEAT: return mPropTable[REQBEHMEM_REPEAT_NEXT];
  case PCIE_REQBEH_LEN: return mPropTable[REQBEHMEM_LENGTH];
  case PCIE_REQBEH_REPLACE_STP: return mPropTable[REQBEHMEM_REPLACE_STP];
  case PCIE_REQBEH_REPLACE_END: return mPropTable[REQBEHMEM_REPLACE_END];
  case PCIE_REQBEH_REPLACE_SEQNO: return mPropTable[REQBEHMEM_REPLACE_SEQNO];
  default:
    AGT_THROW("Invalid CPCIEMemInfo EPCIEReqBeh property");
  }
}

//---------------------------------------------------------------------------
void
CPCIEMemInfo::gapWrite(const UInt32& val, UInt32& lowVal, UInt32& hiVal)
{
  writeSplitVal(val,lowVal,hiVal,REQBEHMEM_GAP_12_0,REQBEHMEM_GAP_15_13);
}

//---------------------------------------------------------------------------
UInt32 
CPCIEMemInfo::gapRead(const UInt32& lowVal, const UInt32& hiVal)
{
  return readSplitVal(lowVal,hiVal,REQBEHMEM_GAP_12_0,REQBEHMEM_GAP_15_13);
}

//---------------------------------------------------------------------------
void
CPCIEMemInfo::gapDefaultWrite(UInt32& lowVal, UInt32& hiVal)
{
  splitValWriteDefault(lowVal,hiVal,REQBEHMEM_GAP_12_0,REQBEHMEM_GAP_15_13);
}

// ----------------------------------------------------------------------- 
// Completer Behaviour Memory
// ----------------------------------------------------------------------- 
UInt8 
CPCIEMemInfo::getBitPos(const EPCIECompBeh& prop)
{
  return getPropertyInfo(prop).mBitPos;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIEMemInfo::getBitSize(const EPCIECompBeh& prop)
{
  return getPropertyInfo(prop).mBitSize;
}

// ----------------------------------------------------------------------- 
UInt32 
CPCIEMemInfo::getDefault(const EPCIECompBeh& prop)
{
  return getPropertyInfo(prop).mDefault;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIEMemInfo::getDWNum(const EPCIECompBeh& prop)
{
  return getPropertyInfo(prop).mDwNum;
}


//---------------------------------------------------------------------------
void 
CPCIEMemInfo::setPropVal
(
 const EPCIECompBeh& prop,
 UInt32& currentVal,
 const UInt32& newOffsetValue 
)
{
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitPos(prop);

  setBits(currentVal,bitPos,bitSize,newOffsetValue);
}

//---------------------------------------------------------------------------
void 
CPCIEMemInfo::setPropDefaultVal
(
 const EPCIECompBeh& prop,
 UInt32& currentVal
)
{
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitPos(prop);
  UInt32 newOffsetValue = getDefault(prop);

  setBits(currentVal,bitPos,bitSize,newOffsetValue);
}


//---------------------------------------------------------------------------
UInt32
CPCIEMemInfo::getPropVal
(
 const EPCIECompBeh& prop,
 const UInt32& currentVal
)
{
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitPos(prop);
  return getBits(currentVal,bitPos,bitSize);

}
// ----------------------------------------------------------------------- 
CPCIEMemInfoProperty 
CPCIEMemInfo::getPropertyInfo(const EPCIECompBeh& prop)
{
  switch(prop)
  {
  case PCIE_COMPBEH_EP: return mPropTable[COMPBEHMEM_EP];
  case PCIE_COMPBEH_TD: return mPropTable[COMPBEHMEM_TD];
  case PCIE_COMPBEH_INCORRECT_LCRC: return mPropTable[COMPBEHMEM_INCORRECT_LCRC];
  case PCIE_COMPBEH_NULLIFIED_TLP: return mPropTable[COMPBEHMEM_NULLIFIED_TLP];
  case PCIE_COMPBEH_WRONG_PYLD_SIZE: return mPropTable[COMPBEHMEM_WRONG_PYLD_SIZE];
  case PCIE_COMPBEH_INSERT_TD: return mPropTable[COMPBEHMEM_INSERT_TD];
  case PCIE_COMPBEH_PRIORITY: return mPropTable[COMPBEHMEM_PRIORITY];
  case PCIE_COMPBEH_INCORRECT_ECRC: return mPropTable[COMPBEHMEM_INCORRECT_ECRC];
  case PCIE_COMPBEH_INCORRECT_DISP: return mPropTable[COMPBEHMEM_INCORRECT_DISP];
  case PCIE_COMPBEH_COMPSTATUS: return mPropTable[COMPBEHMEM_COMPL_STATUS];
  case PCIE_COMPBEH_RCB_COUNT: return mPropTable[COMPBEHMEM_RCBCOUNT];
  //  case PCIE_COMPBEH_VALID: return mPropTable[];
  //  case PCIE_COMPBEH_RESTART: return mPropTable[];
  //  case PCIE_COMPBEH_NOF_ENTRIES: return mPropTable[];
  case PCIE_COMPBEH_REPLACE_STP: return mPropTable[COMPBEHMEM_REPLACE_STP];
  case PCIE_COMPBEH_REPLACE_END: return mPropTable[COMPBEHMEM_REPLACE_END];
  case PCIE_COMPBEH_REPLACE_SEQNO: return mPropTable[COMPBEHMEM_REPLACE_SEQNO];
  case PCIE_COMPBEH_DISCARD: return mPropTable[COMPBEHMEM_DISCARD];
  case PCIE_COMPBEH_REPEAT: return mPropTable[COMPBEHMEM_REPEAT_NEXT];
  default:
    AGT_THROW("Invalid CPCIEMemInfo EPCIECompBeh property");
  }
}

// ----------------------------------------------------------------------- 
// Helper functions 
//---------------------------------------------------------------------------
UInt32
CPCIEMemInfo::getBits
(
  const UInt32& val,
  const UInt8& bitPos,
  const UInt8& bitSize
)
{
  UInt32 mask;
  if(bitSize == 32)
  {
    return val;
  }
  else
  {
    mask = ((0x1 << bitSize) - 1);
    return (val >> bitPos) & mask;
  }
}

//---------------------------------------------------------------------------
void 
CPCIEMemInfo::setBits
(
  UInt32& val,
  const UInt8& bitPos,
  const UInt8& bitSize,
  const UInt32& bitVal
)
{
  UInt32 mask;
  if(bitSize == 32)
  {
    val = bitVal;
  }
  else
  {
    mask = ((0x1 << bitSize) - 1);
    val &= ~(mask << bitPos);
    val |=  ((mask & bitVal) << bitPos);
  }
}

//---------------------------------------------------------------------------
UInt32 
CPCIEMemInfo::readSplitVal
(
 const UInt32& lowVal, 
 const UInt32& hiVal,
 const MemProperty& lowProp, 
 const MemProperty& hiProp
)
{
  UInt32 tmp = 0x0;
  
  UInt8 bitPos = mPropTable[lowProp].mBitPos;
  UInt8 bitSize1 = mPropTable[lowProp].mBitSize;

  tmp |= getBits(lowVal,bitPos,bitSize1);

  bitPos = mPropTable[hiProp].mBitPos;
  UInt8 bitSize2 = mPropTable[hiProp].mBitSize;

  tmp |= getBits(hiVal,bitPos,bitSize2) << bitSize1;

  return tmp;
}

//---------------------------------------------------------------------------
void
CPCIEMemInfo::writeSplitVal
(
 const UInt32& val, 
 UInt32& lowVal, 
 UInt32& hiVal,
 const MemProperty& lowProp, 
 const MemProperty& hiProp
 )
{
  UInt8 bitPos = mPropTable[lowProp].mBitPos;
  UInt8 bitSize1 = mPropTable[lowProp].mBitSize;
  UInt32 bitVal = val & ((0x1 << bitSize1)-1);

  setBits(lowVal,bitPos,bitSize1,bitVal);

  bitPos = mPropTable[hiProp].mBitPos;
  UInt8 bitSize2 = mPropTable[hiProp].mBitSize;
  bitVal = (val >> bitSize1) & ((0x1 << bitSize2)-1);

  setBits(hiVal,bitPos,bitSize2,bitVal);
}

//---------------------------------------------------------------------------
void
CPCIEMemInfo::splitValWriteDefault
(
 UInt32& lowVal, 
 UInt32& hiVal,
 const MemProperty& lowProp,
 const MemProperty& hiProp
 )
{
  UInt8 bitSize = mPropTable[lowProp].mBitSize;
  UInt32 val = mPropTable[lowProp].mDefault | (mPropTable[hiProp].mDefault << bitSize);

  writeSplitVal(val, lowVal, hiVal,lowProp,hiProp);
}
